<?php

namespace WP_SMS\Gateway;

if (!defined('ABSPATH')) exit; // Exit if accessed directly

class unifonic extends \WP_SMS\Gateway
{
    private $wsdl_link = "https://basic.unifonic.com/rest/";
    public $tariff = "https://www.unifonic.com/";
    public $unitrial = false;
    public $unit;
    public $flash = "disable";
    public $isflash = false;

    public function __construct()
    {
        parent::__construct();
        $this->bulk_send      = true;
        $this->has_key        = true;
        $this->gatewayFields  = [
            'username' => [
                'id'           => 'gateway_username',
                'name'         => 'API Username',
                'place_holder' => 'e.g., YourGatewayUsername123',
                'desc'         => 'Enter the username provided by your SMS gateway.',
            ],
            'password' => [
                'id'           => 'gateway_password',
                'name'         => 'API Password',
                'place_holder' => 'e.g., YourGatewayPassword456',
                'desc'         => 'Enter the password associated with your SMS gateway account.',
            ],
            'from'     => [
                'id'           => 'gateway_sender_id',
                'name'         => 'Sender Number',
                'place_holder' => 'e.g., +1 555 123 4567',
                'desc'         => 'This is the number or sender ID displayed on recipients’ devices.
It might be a phone number (e.g., +1 555 123 4567) or an alphanumeric ID if supported by your gateway.',
            ],
            'has_key'  => [
                'id'   => 'gateway_key',
                'name' => 'AppSid',
                'desc' => 'Enter AppSid token of gateway. Check link: <a href="https://software.unifonic.com/en/devtools/restApp" target="_blank">Click Here</a>'
            ]
        ];
        $this->validateNumber = "e.g. 96655xxxxxxx, 96655xxxxxxx";
    }

    public function SendSMS()
    {

        /**
         * Modify sender number
         *
         * @param string $this ->from sender number.
         *
         * @since 3.4
         *
         */
        $this->from = apply_filters('wp_sms_from', $this->from);

        /**
         * Modify Receiver number
         *
         * @param array $this ->to receiver number
         *
         * @since 3.4
         *
         */
        $this->to = apply_filters('wp_sms_to', $this->to);

        /**
         * Modify text message
         *
         * @param string $this ->msg text message.
         *
         * @since 3.4
         *
         */
        $this->msg = apply_filters('wp_sms_msg', $this->msg);

        // Get the credit.
        $credit = $this->GetCredit();

        // Check gateway credit
        if (is_wp_error($credit)) {
            // Log the result
            $this->log($this->from, $this->msg, $this->to, $credit->get_error_message(), 'error');

            return $credit;
        }

        $country_code = isset($this->options['mobile_county_code']) ? $this->options['mobile_county_code'] : '';

        foreach ($this->to as $number) {
            $to       = $this->clean_number($number, $country_code);
            $response = wp_remote_post($this->wsdl_link . 'SMS/messages', [
                'headers' => [
                    'Authorization' => 'Basic ' . base64_encode($this->username . ':' . $this->password),
                    'Accept'        => 'application/json',
                ],
                'body'    => [
                    'AppSid'    => $this->has_key,
                    'SenderID'  => $this->from,
                    'Recipient' => $to,
                    'Body'      => $this->msg,
                ]
            ]);
        }

        // Check gateway response
        if (is_wp_error($response)) {
            // Log the result
            $this->log($this->from, $this->msg, $this->to, $response->get_error_message(), 'error');

            return new \WP_Error('send-sms', $response->get_error_message());
        }

        $response_code = wp_remote_retrieve_response_code($response);

        if ($response_code == '200') {
            $result = json_decode($response['body']);

            if (isset($result->success) and $result->success == 'true') {

                // Log the result
                $this->log($this->from, $this->msg, $this->to, $result);

                /**
                 * Run hook after send sms.
                 *
                 * @param string $result result output.
                 *
                 * @since 2.4
                 *
                 */
                do_action('wp_sms_send', $result);

                return $result;
            } else {
                // Log the result
                $this->log($this->from, $this->msg, $this->to, $response['body'], 'error');

                return new \WP_Error('send-sms', $response['body']);
            }

        } else {
            // Log the result
            $this->log($this->from, $this->msg, $this->to, $response['body'], 'error');

            return new \WP_Error('send-sms', $response['body']);
        }
    }

    public function GetCredit()
    {
        return true;
    }

    /**
     * Clean number
     *
     * @param $number
     *
     * @return bool|string
     */
    private function clean_number($number, $country_code)
    {
        //Clean Country Code from + or 00
        $country_code = str_replace('+', '', $country_code);

        if (substr($country_code, 0, 2) == "00") {
            $country_code = substr($country_code, 2, strlen($country_code));
        }

        //Remove +
        $number = str_replace('+', '', $number);

        if (substr($number, 0, strlen($country_code) * 2) == $country_code . $country_code) {
            $number = substr($number, strlen($country_code) * 2);
        } else {
            $number = substr($number, strlen($country_code));
        }

        //Remove 00 in the begining
        if (substr($number, 0, 2) == "00") {
            $number = substr($number, 2, strlen($number));
        }

        //Remove 00 in the begining
        if (substr($number, 0, 1) == "0") {
            $number = substr($number, 1, strlen($number));
        }

        return $country_code . $number;
    }
}